/* SPDX-License-Identifier: MIT */

/*
 * On Linux, Steam periodically calls SDL_DisableScreenSaver() so your
 * screensaver doesn't work with the Steam client open even if you aren't
 * playing a game, as described in
 * https://github.com/ValveSoftware/steam-for-linux/issues/5607 .
 *
 * To fix this, LD_PRELOAD a library that replaces SDL_DisableScreenSaver()
 * with a no-op if the executable calling it is Steam, but otherwise let it
 * through for other applications. (And print some messages for debugging.)
 *
 * Compile this file with
 *     gcc -shared -fPIC -ldl -m32 -o fix_steam_screensaver_lib.so fix_steam_screensaver.c
 *     gcc -shared -fPIC -ldl -m64 -o fix_steam_screensaver_lib64.so fix_steam_screensaver.c
 *
 * and launch Steam with
 *     LD_PRELOAD="fix_steam_screensaver_\$LIB.so"
 * .
 * 
 * -- This source file has been modified by Jordi Bosveld <jordi@jotti.nl>
 * -- Please do not bother the person(s) mentioned in the github URL with
 * -- questions and/or bug reports.
 */

#define _GNU_SOURCE /* RTLD_NEXT is a GNU extension. */
#include <dlfcn.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#ifndef __linux__
#error Platform not supported.
#endif

#ifdef __i386__
#define ARCH "i386"
#elif defined(__amd64__)
#define ARCH "amd64"
#else
#error Architecture not supported.
#endif

static char exe_name[1024];
static void (*real_function)(void);

static void vlog(const char* text)
{
    fprintf(stderr, "[" ARCH "] %s: %s\n", exe_name, text);
}

static void call_real_function(void)
{
    if (real_function == NULL) {
        void *sdl;

        sdl = dlopen("libSDL2-2.0.so.0", RTLD_NOW | RTLD_GLOBAL);
        if (sdl != NULL) {
            /* Thankfully SDL_DisableScreenSaver() only exists since SDL
             * 2.0, else I'd have to detect the SDL version. */
            real_function = dlsym(sdl, "SDL_DisableScreenSaver");
        }

        if (real_function != NULL)
            vlog("Successfully linked SDL_DisableScreenSaver().");
    }

    if (real_function != NULL) {
        vlog("Allowing SDL_DisableScreenSaver().");
        real_function();
    } else
        vlog("Could not link SDL_DisableScreenSaver().");
}

static int is_steam(void)
{
    static int inited;
    static int retval;

    if (inited)
        return retval;
    inited = 1;

    retval = strcmp(exe_name, "steam") == 0;

    return retval;
}

void SDL_DisableScreenSaver(void)
{
    if (exe_name[0] == '\0') {
        FILE *fp = fopen("/proc/self/comm", "r");

        if (fp != NULL && fgets(exe_name, sizeof(exe_name), fp) && strlen(exe_name) > 0) {
            // strip the newline
            exe_name[strlen(exe_name) - 1] = '\0';
        } else {
            strcpy(exe_name, "(unknown)");
        }

        if (fp != NULL) fclose(fp);
    }

    if (is_steam())
        vlog("Prevented SDL_DisableScreenSaver().");
    else
        call_real_function();
}
